// Copyright 1998-2015 Epic Games, Inc. All Rights Reserved.

#include "PACKT_CPP.h"
#include "PACKT_CPPCharacter.h"

//////////////////////////////////////////////////////////////////////////
// APACKT_CPPCharacter

APACKT_CPPCharacter::APACKT_CPPCharacter()
{
	// Set size for collision capsule
	GetCapsuleComponent()->InitCapsuleSize(42.f, 96.0f);

	// set our turn rates for input
	BaseTurnRate = 45.f;
	BaseLookUpRate = 45.f;

	// Don't rotate when the controller rotates. Let that just affect the camera.
	bUseControllerRotationPitch = false;
	bUseControllerRotationYaw = false;
	bUseControllerRotationRoll = false;

	// Configure character movement
	GetCharacterMovement()->bOrientRotationToMovement = true; // Character moves in the direction of input...	
	GetCharacterMovement()->RotationRate = FRotator(0.0f, 540.0f, 0.0f); // ...at this rotation rate
	GetCharacterMovement()->JumpZVelocity = 600.f;
	GetCharacterMovement()->AirControl = 0.2f;

	// Create a camera boom (pulls in towards the player if there is a collision)
	CameraBoom = CreateDefaultSubobject<USpringArmComponent>(TEXT("CameraBoom"));
	CameraBoom->AttachTo(RootComponent);
	CameraBoom->TargetArmLength = 300.0f; // The camera follows at this distance behind the character	
	CameraBoom->bUsePawnControlRotation = true; // Rotate the arm based on the controller

	// Create a follow camera
	FollowCamera = CreateDefaultSubobject<UCameraComponent>(TEXT("FollowCamera"));
	FollowCamera->AttachTo(CameraBoom, USpringArmComponent::SocketName); // Attach the camera to the end of the boom and let the boom adjust to match the controller orientation
	FollowCamera->bUsePawnControlRotation = false; // Camera does not rotate relative to arm

	Health = 100.f;
	InitialDelay = 2.f;
	RegenDelay = 0.5;
	RegenerateAmount = 1.f;

	// Note: The skeletal mesh and anim blueprint references on the Mesh component (inherited from Character) 
	// are set in the derived blueprint asset named MyCharacter (to avoid direct content references in C++)
}

//////////////////////////////////////////////////////////////////////////
// Input

void APACKT_CPPCharacter::SetupPlayerInputComponent(class UInputComponent* InputComponent)
{
	// Set up gameplay key bindings
	check(InputComponent);
	InputComponent->BindAction("Jump", IE_Pressed, this, &ACharacter::Jump);
	InputComponent->BindAction("Jump", IE_Released, this, &ACharacter::StopJumping);

	InputComponent->BindAxis("MoveForward", this, &APACKT_CPPCharacter::MoveForward);
	InputComponent->BindAxis("MoveRight", this, &APACKT_CPPCharacter::MoveRight);

	// We have 2 versions of the rotation bindings to handle different kinds of devices differently
	// "turn" handles devices that provide an absolute delta, such as a mouse.
	// "turnrate" is for devices that we choose to treat as a rate of change, such as an analog joystick
	InputComponent->BindAxis("Turn", this, &APawn::AddControllerYawInput);
	InputComponent->BindAxis("TurnRate", this, &APACKT_CPPCharacter::TurnAtRate);
	InputComponent->BindAxis("LookUp", this, &APawn::AddControllerPitchInput);
	InputComponent->BindAxis("LookUpRate", this, &APACKT_CPPCharacter::LookUpAtRate);

	// handle touch devices
	InputComponent->BindTouch(IE_Pressed, this, &APACKT_CPPCharacter::TouchStarted);
	InputComponent->BindTouch(IE_Released, this, &APACKT_CPPCharacter::TouchStopped);
}


float APACKT_CPPCharacter::TakeDamage( float Damage, struct FDamageEvent const& DamageEvent, AController* EventInstigator, AActor* DamageCauser )
{
	// Get the actual damage applied
	const float ActualDamage = Super::TakeDamage(Damage, DamageEvent, EventInstigator, DamageCauser);

	if (Health <= 0.0)
	{
		// Player has no health. So return 0.0
		return 0.0;
	}

	// Reduce health by the damage received
	Health = Health - ActualDamage;

	//Is the health reduced to 0 for the first time?
	if (Health <= 0.0)
	{
		// Clear existing timer
		GetWorldTimerManager().ClearTimer(TimerHandle_HealthRegen);

		// Call the BLueprint event
		PlayerHealthIsZero();

		return 0.0;
	}


	// Set a timer to call Regenerate Health function, if it is not running already
	if (!GetWorldTimerManager().IsTimerActive(TimerHandle_HealthRegen))
	{
		GetWorldTimerManager().SetTimer(TimerHandle_HealthRegen, this, &APACKT_CPPCharacter::RegenerateHealth, InitialDelay);
	}

	// return the actual damage received
	return ActualDamage;
}

void APACKT_CPPCharacter::RegenerateHealth()
{
	if (Health >= GetClass()->GetDefaultObject<ABaseCharacter>()->Health)
	{
		Health = GetClass()->GetDefaultObject<ABaseCharacter>()->Health;
	}
	else
	{
		Health += RegenerateAmount;
		FTimerHandle TimerHandle_ReRunRegenerateHealth;
		GetWorldTimerManager().SetTimer(TimerHandle_ReRunRegenerateHealth, this, &APACKT_CPPCharacter::RegenerateHealth, RegenDelay);
	}
}

void APACKT_CPPCharacter::TouchStarted( ETouchIndex::Type FingerIndex, FVector Location )
{
	// jump, but only on the first touch
	if (FingerIndex == ETouchIndex::Touch1)
	{
		Jump();
	}
}

void APACKT_CPPCharacter::TouchStopped(ETouchIndex::Type FingerIndex, FVector Location)
{
	if (FingerIndex == ETouchIndex::Touch1)
	{
		StopJumping();
	}
}

void APACKT_CPPCharacter::TurnAtRate(float Rate)
{
	// calculate delta for this frame from the rate information
	AddControllerYawInput(Rate * BaseTurnRate * GetWorld()->GetDeltaSeconds());
}

void APACKT_CPPCharacter::LookUpAtRate(float Rate)
{
	// calculate delta for this frame from the rate information
	AddControllerPitchInput(Rate * BaseLookUpRate * GetWorld()->GetDeltaSeconds());
}

void APACKT_CPPCharacter::MoveForward(float Value)
{
	if ((Controller != NULL) && (Value != 0.0f))
	{
		// find out which way is forward
		const FRotator Rotation = Controller->GetControlRotation();
		const FRotator YawRotation(0, Rotation.Yaw, 0);

		// get forward vector
		const FVector Direction = FRotationMatrix(YawRotation).GetUnitAxis(EAxis::X);
		AddMovementInput(Direction, Value);
	}
}

void APACKT_CPPCharacter::MoveRight(float Value)
{
	if ( (Controller != NULL) && (Value != 0.0f) )
	{
		// find out which way is right
		const FRotator Rotation = Controller->GetControlRotation();
		const FRotator YawRotation(0, Rotation.Yaw, 0);
	
		// get right vector 
		const FVector Direction = FRotationMatrix(YawRotation).GetUnitAxis(EAxis::Y);
		// add movement in that direction
		AddMovementInput(Direction, Value);
	}
}
